﻿/*  Author:     Ben Hall
 *  File:       HealerAI.cs
 *  Date:       4/28/2011
 *  
 *  Purpose:    HealerAI.cs extends the AICharacter class and overrides the abstract method MakeMove()
 *              using a decision tree that focuses on healing itself or allies, or using defensive skills
 *              on itself or its allies.  It is less likely to attack an opponent than any other AICharacter
 *              type.
 *              
 *              The MakeMove() method also uses a random chance to skip the "best" move the AICharacter
 *              can make at each branch, based on the AICharacter's intelligence, with the chance of it
 *              skipping a "best" move decreasing every time the "best" move is skipped.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace OnlineGame
{
    public class HealerAI : AICharacter
    {
        public HealerAI(String name, int strIn, int stmIn, int aglIn, int intIn, int fthIn, int prsIn, int currencyIn)
        {
            this.characterName = name;
            this.strength = new CharStat(strIn);
            this.stamina = new CharStat(stmIn);
            this.agility = new CharStat(aglIn);
            this.intelligence = new CharStat(intIn);
            this.faith = new CharStat(fthIn);
            this.persona = new CharStat(prsIn);
            //this.level = DetermineLevel();

            this.currencyCarried = currencyIn;

            this.maxHealth = DetermineMaxHealth();
            this.currentHealth = maxHealth;
            this.maxMagic = DetermineMaxMagic();
            this.currentMagic = maxMagic;

            this.defending = false;
            this.poisoned = false;
            this.covering = false;
            this.coveredBy = null;
            this.characterCovered = null;

            this.skillsKnown = new bool[MainPage.skillList.Length];
            for (int i = 0; i < skillsKnown.Length; i++)
            {
                skillsKnown[i] = false;
            }

            this.decisionChance = DetermineDecisionChance();
        }

        /*  MakeMove() selects an action for the AICharacter to take, using a decision tree with a
         *  random chance of skipping the "best move" at each branch (based on the AICharacter's intelligence
         *  score)
         *  
         *  HealerAI prioritizes healing itself and its allies, as well as boosting their abilities,
         *  before attacking.
         */
        public override String MakeMove()
        {
            String messageReturned = "";
            this.decisionChance = DetermineDecisionChance();
            this.rndDecision = MainPage.rnd.NextDouble();
            //AI may decide to skip more optimal moves, with lower intelligence being more likely to skip moves
            bool useBestMove = false;
            useBestMove = (rndDecision < decisionChance);

            int chosenSkill = GetCompHealSkill(); // will return -1 if no heal skill is available for use
            if ((this.GetHealthPercent() < lowHealth) && (chosenSkill >= 0) && (useBestMove))
            {
                //heal self with best healing skill if health is below 25%
                messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, this);
            }
            else
            {
                if (!useBestMove)
                {
                    useBestMove = RedetermineUseBestMove();
                }
                //attempt to find an ally at lowHealth and find the lowest health among those.
                AICharacter healTarget = null;
                for (int i = 0; i < MainPage.compTeam.Length; i++)
                {
                    if ((MainPage.compTeam[i].GetCurrentHealth() > 0) && (!MainPage.compTeam[i].Equals(this)) && (MainPage.compTeam[i].GetHealthPercent() < lowHealth))
                    {
                        if (healTarget == null)
                        {
                            healTarget = MainPage.compTeam[i];
                        }
                        else if (healTarget.GetCurrentHealth() > MainPage.compTeam[i].GetCurrentHealth())
                        {
                            healTarget = MainPage.compTeam[i];
                        }
                    }
                }
                chosenSkill = GetCompHealSkill();
                if ((healTarget != null) && (chosenSkill >= 0) && (useBestMove))
                {
                    //heal target
                    messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, healTarget);
                }
                else
                {
                    //redetermine useBestMove
                    if (!useBestMove)
                    {
                        useBestMove = RedetermineUseBestMove();
                    }
                    //see if any are poisoned, if so heal the one with lowest health%
                    healTarget = null;
                    for (int i = 0; i < MainPage.compTeam.Length; i++)
                    {
                        if (MainPage.compTeam[i].IsPoisoned())
                        {
                            if (healTarget == null)
                            {
                                healTarget = MainPage.compTeam[i];
                            }
                            else if (healTarget.GetHealthPercent() > MainPage.compTeam[i].GetHealthPercent())
                            {
                                healTarget = MainPage.compTeam[i];
                            }
                        }
                    }
                    chosenSkill = GetCompRemovePoisonSkill();
                    if ((healTarget != null) && (chosenSkill >= 0) && (useBestMove))
                    {
                        //cure poison
                        messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, healTarget);
                    }
                    else
                    {
                        //redetermine useBestMove
                        if (!useBestMove)
                        {
                            useBestMove = RedetermineUseBestMove();
                        }
                        bool anyCompUnderHighHealth = false;
                        for (int i = 0; i < MainPage.compTeam.Length; i++)
                        {
                            //determine if any living AICharacters are under 75% health
                            if ((MainPage.compTeam[i].GetCurrentHealth() > 0) && (MainPage.compTeam[i].GetHealthPercent() < highHealth))
                            {
                                anyCompUnderHighHealth = true;
                            }
                        }

                        if ((anyCompUnderHighHealth) && (useBestMove))
                        {
                            //if able to heal: 50% chance of healing lowest health%, 30% of Defensive action, 10% Fight, 10% chance of Debuff
                            //else 50% chance of Defensive, 20% chance of Debuff, 30% chance of Attack
                            chosenSkill = GetCompHealSkill();
                            if (chosenSkill >= 0)
                            {
                                double random = MainPage.rnd.NextDouble();
                                if (random < 0.5)
                                {
                                    healTarget = null;
                                    for (int i = 0; i < MainPage.compTeam.Length; i++)
                                    {
                                        //target only living comps
                                        if (MainPage.compTeam[i].GetCurrentHealth() > 0)
                                        {
                                            //find comp with lowest health
                                            if (healTarget == null)
                                            {
                                                healTarget = MainPage.compTeam[i];
                                            }
                                            else if (healTarget.GetHealthPercent() > MainPage.compTeam[i].GetHealthPercent())
                                            {
                                                healTarget = MainPage.compTeam[i];
                                            }
                                        }
                                    }
                                    messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, healTarget);
                                }
                                else if (random < 0.8)
                                {
                                    //Defensive action
                                    messageReturned = UseBestDefense();
                                }
                                else if (random < 0.9)
                                {
                                    //Fight
                                    messageReturned = MainPage.FightCommand(this, MainPage.playerCharacter);
                                }
                                else
                                {
                                    messageReturned = UseRandomDebuff();
                                }
                            }
                        }
                        else
                        {
                            //50% chance of Defensive action, 20% chance of Debuff action, 30% chance of Attack action
                            double random = MainPage.rnd.NextDouble();
                            if (random < 0.5)
                            {
                                messageReturned = UseBestDefense();
                            }
                            else if (random < 0.7)
                            {
                                messageReturned = UseRandomDebuff();
                            }
                            else
                            {
                                messageReturned = UseBestAttack();
                            }
                        }
                    }
                }
            }
            return messageReturned;
        }

    }
}
